<?php
/**
 * 🏠 LANDING PAGE - MAIN ENTRY POINT
 * =================================
 * 
 * This is the main landing page (index.php) that displays multilingual content.
 * Previously this was prescription.php, but was renamed to be the main entry point.
 * 
 * 📋 WHAT THIS FILE DOES:
 * - Displays customizable multilingual landing page content
 * - Auto-detects user's browser language (English/French)
 * - Processes special syntax for images and download links
 * - Provides fallback content if landing.php is missing
 * - Includes navigation to all other site sections
 * 
 * 🔧 CUSTOMIZATION:
 * - Edit landing.php to change the content
 * - Edit landing.php to change site name and colors
 * - Content supports HTML, images, and download links
 * 
 * 🌐 MULTILINGUAL SUPPORT:
 * - Content managed in landing.php (separate from UI text)
 * - UI text settings managed in settings.php
 * - Auto-detection based on browser language
 */

// =============================================================================
// 📦 INCLUDE REQUIRED FILES
// =============================================================================
require_once 'settings.php';      // Main configuration and translations
require_once 'user_functions.php'; // User management and utility functions
require_once 'landing.php';       // Landing page content (multilingual)

// =============================================================================
// 🔐 SESSION MANAGEMENT
// =============================================================================
session_start(); // Start PHP session for user login state and language preference

// =============================================================================
// 🌐 LANGUAGE DETECTION AND CONTENT LOADING
// =============================================================================

// Get current language (auto-detected from browser or session)
$lang = getCurrentLanguage();

// Load landing page content for the detected/selected language
$landing_data = getLandingContent($lang);
$landing_title = $landing_data['title'] ?? 'Welcome';           // Page title
$landing_content = processLandingContent($landing_data['content'] ?? []); // Processed HTML content

// =============================================================================
// 🛡️ FALLBACK CONTENT HANDLING
// =============================================================================

// If no content is available (missing landing.php or empty content), show fallback
if (empty($landing_content)) {
    $landing_content = '<p>Welcome to ' . getSetting('SITE_SETTINGS.site_name', 'Principles') . '!</p><p>Edit landing.php to customize this content.</p>';
}

?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo getSetting('SITE_SETTINGS.site_name', 'Principles'); ?> - <?php echo t('prescription'); ?></title>
    <link rel="stylesheet" href="style.css">
    <link rel="stylesheet" href="dynamic-styles.php">
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar">
        <div class="nav-container nav-flex">
            <div class="nav-left">
                <a href="index.php" class="nav-button"><?php echo t('prescription', $lang); ?></a>
                <a href="principles.php" class="nav-button"><?php echo t('principles', $lang); ?></a>
                <a href="walkers.php" class="nav-button"><?php echo t('walkers', $lang); ?></a>
                <?php if (isLoggedIn()): ?>
                    <a href="profile.php" class="nav-button profile-btn"><?php echo t('profile', $lang); ?></a>
                <?php else: ?>
                    <a href="signup.php" class="nav-button login-btn"><?php echo t('login_signup', $lang); ?></a>
                <?php endif; ?>
            </div>
        </div>
    </nav>

    <!-- Main Content -->
    <main class="main-content">
        <div class="container">
            <div class="prescription-content">
                <h1 class="landing-title"><?php echo htmlspecialchars($landing_title); ?></h1>
                <?php echo $landing_content; ?>
            </div>
        </div>
    </main>
</body>
</html>
