<?php
/**
 * 🌐 LANDING PAGE CONTENT - BILINGUAL CONTENT MANAGER
 * ===================================================
 * 
 * This file contains all homepage/prescription content for your bilingual website.
 * It works alongside settings.php but focuses specifically on landing page content.
 * 
 * 📝 WHAT YOU CAN CUSTOMIZE:
 * - Landing page content in English and one other language
 * - Images using IMAGE: syntax
 * - Project download links using LINK: syntax
 * - Rich text formatting with HTML tags
 * - Page titles and structured content sections
 * 
 * 🔧 HOW TO USE:
 * 1. Edit the content in the $LANDING_CONTENT array below
 * 2. Save this file
 * 3. Refresh your website to see changes immediately
 * 4. Content automatically displays based on user's browser language
 * 
 * 🌍 STRICT BILINGUAL SYSTEM:
 * - English ('en') is the primary language - always include this
 * - 'other' language matches the setting in settings.php
 * - Browser auto-detection chooses between these two languages only
 * - DO NOT add more languages - system is designed for exactly two
 * - Language detection is consistent with settings.php configuration
 * 
 * 📋 SPECIAL SYNTAX:
 * - IMAGE: [URL] - Displays an image with responsive styling
 * - LINK: [URL] [Text] - Creates a download/project link button
 * - Use \n for line breaks in content
 * - HTML tags are supported for rich formatting
 * - Content is safely processed to handle apostrophes and special characters
 */

// =============================================================================
// 🌐 LANDING PAGE CONTENT IN MULTIPLE LANGUAGES
// =============================================================================

$LANDING_CONTENT = [
    'en' => [
        "title" => '',
        "content" => [
            'IMAGE: https://cdn.pixabay.com/photo/2018/08/04/11/30/draw-3583548_1280.png',
            "I am a doctor, and the prescription I'm sharing helped me heal from difficulties in my life. It is based on pragmatic truth: you apply it for four months (the time it takes for neural reprogramming). If you see improvement, you keep going and it becomes your truth. If it doesn't work, you let it go. It takes 3 minutes. Just like brushing your teeth. You may not know exactly how it works, but if you do it properly, morning and night, cavities disappear. This is the same—only for inner peace.

You start by taking ten deep inhales and ten deep exhales, twice a day: once in the morning, once in the evening. During these breaths, imagine that your skin is you. It is a living barrier. It filters between your outside and your inside. What's inside is a whole, indivisible field. There are no emotions to fix, no thoughts to chase, no inner child to rescue. It's all one mass. You welcome everything. The skin filters what comes in and what goes out. What you say and what you do. The skin is you.

But what do you filter with? With ten principles that you will choose and gradually refine—like cheese or fine wine. These principles are the conductor of your filter. They are the guides you've chosen. And every time you obey one of them, you will have done something right, and it will increase your self-esteem. Which is the foundation for everything else.

Don't change anything in your life, your therapy, or your medication. Just add this, and watch your inner world shift. Watch your outer world shift.

You can do it alone or with us, because the moment you choose your ten principles, you become a walker on the path of your truth—and there is no turning back. This site is here to share that path with others. It helps. Each principle is validated by a human moderator.

And one last thing: your skin and your principles are protected by a word. You choose it. This word becomes your guardian. When you say it—out loud or in silence—your skin becomes firm and impenetrable. It sets a boundary. It expels intrusive thoughts. It protects you from the judgments of others you've believed, and from physical, verbal, emotional, or psychological attacks. Those who don't respect your word are not your friends.

You are the guardian of your inner world. It is whole. And it belongs to you.",
            'LINK: start.zip Download Source Code',
        ]
    ],
    
    'other' => [
        "title" => 'Bienvenue dans Notre Communauté de Principes',
        "content" => [
            'IMAGE: https://cdn.pixabay.com/photo/2018/08/04/11/30/draw-3583548_1280.png',
            "Je suis médecin et la prescription que je partage m'a aidée à surmonter les difficultés de ma vie. Elle repose sur une vérité pragmatique : on l'applique pendant quatre mois (le temps d'une reprogrammation neuronale). Si on constate une amélioration, on persévère et elle devient notre vérité. Si ça ne marche pas, on lâche prise. Cela prend trois minutes. C'est comme se brosser les dents. On ne sait peut-être pas exactement comment ça marche, mais si on le fait correctement, matin et soir, les caries disparaissent. C'est pareil, mais pour la paix intérieure.
            Commencez par dix inspirations et dix expirations profondes, deux fois par jour : une le matin, une le soir. Pendant ces respirations, imaginez que votre peau est vous. C'est une barrière vivante. Elle filtre entre l'extérieur et l'intérieur. Ce qui est à l'intérieur est un champ entier et indivisible. Il n'y a aucune émotion à régler, aucune pensée à chasser, aucun enfant intérieur à sauver. Tout est une seule masse. Vous accueillez tout. La peau filtre ce qui entre et ce qui sort. Ce que vous dites et ce que vous faites. Votre peau, c'est vous.

            Mais avec quoi filtrez-vous ? Avec dix principes que vous choisirez et affinerez progressivement, comme le fromage ou le bon vin. Ces principes sont le guide de votre filtre. Ce sont les guides que vous avez choisis. Et chaque fois que vous en suivrez un, vous aurez fait quelque chose de bien et cela renforcera votre estime de soi. C'est la base de tout le reste.

            Ne changez rien à votre vie, à votre thérapie ou à vos médicaments. Ajoutez simplement ceci et observez votre monde intérieur changer. Observez votre monde extérieur changer.

            Vous pouvez le faire seul ou avec nous, car dès que vous choisissez vos dix principes, vous devenez un marcheur sur le chemin de votre vérité, sans retour en arrière possible. Ce site est là pour partager ce chemin avec d'autres. C'est utile. Chaque principe est validé par un modérateur humain.

            Et une dernière chose : votre peau et vos principes sont protégés par un mot. Vous le choisissez. Ce mot devient votre gardien. Lorsque vous le prononcez, à voix haute ou en silence, votre peau devient ferme et impénétrable. Elle fixe une limite. Elle chasse les pensées intrusives. Elle vous protège des jugements des autres en qui vous avez cru, ainsi que des agressions physiques, verbales, émotionnelles ou psychologiques. Ceux qui ne respectent pas votre parole ne sont pas vos amis.

            Vous êtes le gardien de votre monde intérieur. Il est entier. Et il vous appartient.",
            'LINK: start.zip Télécharger le Code Source',
        ]
    ],
    
    // =============================================================================
    // 🔧 ADD YOUR OWN LANGUAGE HERE
    // =============================================================================
    // Copy the structure above and replace 'xx' with your language code:
    /*
    'xx' => [
        'title' => 'Your Title Here',
        'content' => [
            'Your content line 1',
            'Your content line 2',
            'IMAGE: https://your-image-url.com/image.jpg',
            'LINK: https://your-link.com Your Link Text',
            // Add more content lines as needed
        ]
    ],
    */
];

// =============================================================================
// 🛠️ HELPER FUNCTIONS - DON'T MODIFY BELOW THIS LINE
// =============================================================================

/**
 * Get the current browser language (en or other, defaults to en)
 * This function should be consistent with getCurrentLanguage() in settings.php
 */
function getLandingLanguage() {
    // Include settings to get language configuration
    require_once 'settings.php';
    
    // Check if language is set in session
    if (isset($_SESSION['language'])) {
        return $_SESSION['language'];
    }
    
    // Only auto-detect if enabled in settings
    if (getSetting('LANGUAGES.auto_detect', true)) {
        // Auto-detect browser language using settings configuration
        $browser_lang = $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '';
        $other_lang_code = getSetting('LANGUAGES.other_language_code', 'fr');
        
        if (strpos(strtolower($browser_lang), strtolower($other_lang_code)) !== false) {
            return 'other';
        }
    }
    
    // Return default language from settings
    return getSetting('LANGUAGES.default', 'en');
}

/**
 * Get landing page content for a specific language
 */
function getLandingContent($lang = null) {
    global $LANDING_CONTENT;
    
    if ($lang === null) {
        $lang = getLandingLanguage();
    }
    
    // Return requested language content, fallback to English
    return $LANDING_CONTENT[$lang] ?? $LANDING_CONTENT['en'];
}

/**
 * Process landing content and convert special syntax to HTML
 */
function processLandingContent($content_array) {
    $processed = [];
    
    foreach ($content_array as $line) {
        // Handle IMAGE: syntax
        if (strpos($line, 'IMAGE:') === 0) {
            $url = trim(substr($line, 6));
            if (filter_var($url, FILTER_VALIDATE_URL)) {
                $processed[] = '<div class="landing-image"><img src="' . htmlspecialchars($url) . '" alt="Landing Image" style="max-width: 100%; height: auto; border-radius: 12px; margin: 20px 0;"></div>';
            }
            continue;
        }
        
        // Handle LINK: syntax
        if (strpos($line, 'LINK:') === 0) {
            $link_data = trim(substr($line, 5));
            $parts = explode(' ', $link_data, 2);
            if (count($parts) >= 2) {
                $url = $parts[0];
                $text = $parts[1];
                
                // Handle different URL types
                if (strpos($url, 'http') === 0) {
                    // Already absolute URL
                    $final_url = $url;
                } elseif (strpos($url, '/') === 0) {
                    // Absolute path from domain root
                    $scheme = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://';
                    $host = $_SERVER['HTTP_HOST'];
                    $final_url = $scheme . $host . $url;
                } else {
                    // Relative path - use as is
                    $final_url = $url;
                }
                
                // Create download link (no need to validate relative paths)
                $processed[] = '<a href="' . htmlspecialchars($final_url) . '" class="download-btn" download>' . htmlspecialchars($text) . '</a>';
            }
            continue;
        }
        
        // Handle empty lines (convert to <br>)
        if (trim($line) === '') {
            $processed[] = '<br>';
            continue;
        }
        
        // Handle regular content (safely escape and allow HTML tags)
        $processed[] = '<p>' . htmlspecialchars($line, ENT_QUOTES, 'UTF-8', false) . '</p>';
    }
    
    return implode("\n", $processed);
}

?>
