<?php
/**
 * 👤 USER PROFILE PAGE - COMPREHENSIVE USER MANAGEMENT
 * ==================================================
 * 
 * This file handles all user profile functionality including viewing, editing,
 * and principle management. It supports both public and private profile views.
 * 
 * 📋 MAIN FEATURES:
 * - View user profiles (public and private)
 * - Edit profile information (username, email, password)
 * - Real-time password validation with error messages
 * - Add, edit, and delete personal principles
 * - Moderator controls for principle approval/rejection
 * - Icon-based UI with save/cancel functionality
 * - AJAX updates for smooth user experience
 * - Comprehensive error handling and validation
 * 
 * 🔐 SECURITY FEATURES:
 * - Session-based authentication
 * - User ID validation for profile access
 * - Moderator privilege checking
 * - Password hashing and validation
 * - Input sanitization and validation
 * 
 * 🎨 UI FEATURES:
 * - Mobile-first responsive design
 * - Dynamic color system for principle cards
 * - Icon-based buttons (edit, save, cancel, logout)
 * - Real-time form validation
 * - Success/error message display
 * 
 * 🌐 MULTILINGUAL SUPPORT:
 * - All text and messages from settings.php
 * - Error messages in English and French
 * - Auto-detection of browser language
 */

// =============================================================================
// 📦 INCLUDE REQUIRED FILES AND START SESSION
// =============================================================================
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
require_once 'user_functions.php';

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// =============================================================================
// 🔄 AJAX PROFILE FIELD UPDATES HANDLER
// =============================================================================
// Handle AJAX profile field updates
if (isset($_POST['field']) && isset($_POST['value'])) {
    header('Content-Type: application/json');
    
    if (!isLoggedIn()) {
        echo json_encode(['status' => 'error', 'message' => t('error_not_logged_in', $lang)]);
        exit;
    }
    
    $allowed_fields = ['username', 'email', 'age', 'location', 'password'];
    $field = $_POST['field'] ?? '';
    $value = $_POST['value'] ?? '';
    
    if (!in_array($field, $allowed_fields)) {
        echo json_encode(['status' => 'error', 'message' => t('error_invalid_field', $lang)]);
        exit;
    }
    
    $username = $_SESSION['username'];
    $user = getUserByUsername($username);
    if (!$user) {
        echo json_encode(['status' => 'error', 'message' => t('error_user_not_found', $lang)]);
        exit;
    }
    
    // Direct update without moderation
    $users = getUsers();
    foreach ($users as &$u) {
        if ($u['username'] === $username) {
            if ($field === 'password') {
                $u['password'] = password_hash($value, PASSWORD_DEFAULT);
            } else {
                $u[$field] = $value;
            }
            break;
        }
    }
    saveUsers($users);
    echo json_encode(['status' => 'success', 'message' => '']);
    exit;
}

// Only require login for profile editing, not viewing
if (isset($_POST['logout'])) {
    $current_user = $_SESSION['username'] ?? null;
    logoutUser();
    // Redirect to the same profile in public view
    if ($current_user) {
        header('Location: profile.php?user=' . urlencode($current_user));
    } else {
        header('Location: index.php');
    }
    exit;
}
// Language bar logic
if (isset($_GET['lang']) && in_array($_GET['lang'], ['en','fr'])) {
    $_SESSION['lang'] = $_GET['lang'];
}
$lang = $_SESSION['lang'] ?? get_default_lang();
// Get username from query or session
$username = $_GET['user'] ?? ($_SESSION['username'] ?? null);
if (!$username) {
    header('Location: signup.php?message=' . urlencode(t('error_login_required', $lang)));
    exit;
}
$user = getUserByUsername($username);
if (!$user) {
    // Log out and redirect if user not found
    session_unset();
    session_destroy();
    header('Location: signup.php?message=' . urlencode(t('error_login_again', $lang)));
    exit;
}
$is_own_profile = isset($_SESSION['username']) && $_SESSION['username'] === $user['username'];

// Only allow editing if logged in and it's own profile
if (!$is_own_profile && isset($_POST['username'])) {
    header('Location: signup.php?message=' . urlencode(t('error_edit_permission', $lang)));
    exit;
}

// Get user's principles
$principlesFile = __DIR__ . '/principles.json';
$principles = file_exists($principlesFile) ? json_decode(file_get_contents($principlesFile), true) : [];
if (!is_array($principles)) {
    $principles = [];
}
if ($is_own_profile) {
    $user_principles = array_values(array_filter($principles, function($p) use ($user) {
        return isset($p['posted_by_user_id']) && $p['posted_by_user_id'] === $user['user_id'];
    }));
} else {
    $user_principles = array_values(array_filter($principles, function($p) use ($user) {
        return isset($p['posted_by_user_id']) && 
               $p['posted_by_user_id'] === $user['user_id'] && 
               isset($p['approved']) && 
               $p['approved'] === true;
    }));
}

// Debug output (remove after testing)
// echo "<!-- Debug: User ID: " . $user['user_id'] . ", Is own profile: " . ($is_own_profile ? 'true' : 'false') . ", Principles count: " . count($user_principles) . " -->";

$update_message = '';
if (isset($_POST['username']) && $is_own_profile) {
    $new_username = trim($_POST['username']);
    $new_email = trim($_POST['email'] ?? $user['email']);
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    $users = getUsers();
    foreach ($users as &$u) {
        if ($u['username'] === $user['username']) {
            $u['username'] = $new_username;
            $u['email'] = $new_email;
            if (!empty($new_password)) {
                if ($new_password === $confirm_password && strlen($new_password) >= 6) {
                    $u['password'] = password_hash($new_password, PASSWORD_DEFAULT);
                } else {
                    $update_message = t('error_passwords_mismatch_short', $lang);
                    break;
                }
            }
            break;
        }
    }
    saveUsers($users);
    // Update session username if changed
    if ($_SESSION['username'] !== $new_username) {
        $_SESSION['username'] = $new_username;
    }
    header('Location: profile.php?user=' . urlencode($new_username));
    exit;
}

$principle_message = '';
if (isset($_POST['add_principle']) && $is_own_profile) {
    $title = trim($_POST['principle_title'] ?? '');
    $desc = trim($_POST['principle_desc'] ?? '');
    $titleWords = preg_split('/\\s+/', $title, -1, PREG_SPLIT_NO_EMPTY);
    if ($title && $desc && mb_strlen($desc) <= 777 && count($titleWords) <= 3) {
        $all_principles = file_exists($principlesFile) ? json_decode(file_get_contents($principlesFile), true) : [];
        if (!is_array($all_principles)) {
            $all_principles = [];
        }

        $all_principles[] = [
            'id' => uniqid('p_'),
            'title' => $title,
            'description' => $desc,
            'posted_by_user_id' => $user['user_id'],
            'approved' => false,
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        // Use absolute path for writing
        $result = file_put_contents($principlesFile, json_encode($all_principles, JSON_PRETTY_PRINT));
        if ($result === false) {
            die("FATAL ERROR: Could not write to principles.json. This is a file permission issue. Please double-check that your web server (Apache) has write permissions for the project folder.");
        }

        header('Location: profile.php?user=' . urlencode($user['username']) . '&message=' . urlencode(t('success_principle_submitted', $lang)));
        exit;
    } else {
        $principle_message = t('error_principle_validation', $lang);
    }
}

// Handle edit/delete POST
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['principle_action'])) {
    if ($_POST['principle_action'] === 'delete' && isset($_POST['principle_id'])) {
        $principle_id_to_delete = $_POST['principle_id'];
        $principles = array_filter($principles, function($p) use ($principle_id_to_delete, $user) {
            // Only allow deletion if user owns the principle
            if ($p['id'] === $principle_id_to_delete) {
                return !(isset($p['posted_by_user_id']) && $p['posted_by_user_id'] === $user['user_id']);
            }
            return true;
        });
        file_put_contents($principlesFile, json_encode(array_values($principles), JSON_PRETTY_PRINT));
        header('Location: profile.php?user=' . urlencode($user['username']));
        exit;
    }
    if ($_POST['principle_action'] === 'edit' && isset($_POST['principle_id'], $_POST['edit_title'], $_POST['edit_desc'])) {
        foreach ($principles as &$p) {
            if ($p['id'] === $_POST['principle_id']) {
                // Verify user owns this principle
                if (isset($p['posted_by_user_id']) && $p['posted_by_user_id'] === $user['user_id']) {
                    $p['title'] = $_POST['edit_title'];
                    $p['description'] = $_POST['edit_desc'];
                    $p['approved'] = false; // re-moderate on edit
                }
                break;
            }
        }
        file_put_contents($principlesFile, json_encode(array_values($principles), JSON_PRETTY_PRINT));
        header('Location: profile.php?user=' . urlencode($user['username']));
        exit;
    }
}
// Update user principles after any changes
$user_principles = array_values(array_filter($principles, function($p) use ($user) {
    return isset($p['posted_by_user_id']) && $p['posted_by_user_id'] === $user['user_id'];
}));
$edit_id = $_POST['edit_id'] ?? null;

// Add a country dropdown list
$countries = ["Afghanistan", "Albania", "Algeria", "Andorra", "Angola", "Antigua and Barbuda", "Argentina", "Armenia", "Australia", "Austria", "Azerbaijan", "Bahamas", "Bahrain", "Bangladesh", "Barbados", "Belarus", "Belgium", "Belize", "Benin", "Bhutan", "Bolivia", "Bosnia and Herzegovina", "Botswana", "Brazil", "Brunei", "Bulgaria", "Burkina Faso", "Burundi", "Cabo Verde", "Cambodia", "Cameroon", "Canada", "Central African Republic", "Chad", "Chile", "China", "Colombia", "Comoros", "Congo, Democratic Republic of the", "Congo, Republic of the", "Costa Rica", "Cote d'Ivoire", "Croatia", "Cuba", "Cyprus", "Czech Republic", "Denmark", "Djibouti", "Dominica", "Dominican Republic", "Ecuador", "Egypt", "El Salvador", "Equatorial Guinea", "Eritrea", "Estonia", "Eswatini", "Ethiopia", "Fiji", "Finland", "France", "Gabon", "Gambia", "Georgia", "Germany", "Ghana", "Greece", "Grenada", "Guatemala", "Guinea", "Guinea-Bissau", "Guyana", "Haiti", "Honduras", "Hungary", "Iceland", "India", "Indonesia", "Iran", "Iraq", "Ireland", "Israel", "Italy", "Jamaica", "Japan", "Jordan", "Kazakhstan", "Kenya", "Kiribati", "Kosovo", "Kuwait", "Kyrgyzstan", "Laos", "Latvia", "Lebanon", "Lesotho", "Liberia", "Libya", "Liechtenstein", "Lithuania", "Luxembourg", "Madagascar", "Malawi", "Malaysia", "Maldives", "Mali", "Malta", "Marshall Islands", "Mauritania", "Mauritius", "Mexico", "Micronesia", "Moldova", "Monaco", "Mongolia", "Montenegro", "Morocco", "Mozambique", "Myanmar", "Namibia", "Nauru", "Nepal", "Netherlands", "New Zealand", "Nicaragua", "Niger", "Nigeria", "North Korea", "North Macedonia", "Norway", "Oman", "Pakistan", "Palau", "Palestine", "Panama", "Papua New Guinea", "Paraguay", "Peru", "Philippines", "Poland", "Portugal", "Qatar", "Romania", "Russia", "Rwanda", "Saint Kitts and Nevis", "Saint Lucia", "Saint Vincent and the Grenadines", "Samoa", "San Marino", "Sao Tome and Principe", "Saudi Arabia", "Senegal", "Serbia", "Seychelles", "Sierra Leone", "Singapore", "Slovakia", "Slovenia", "Solomon Islands", "Somalia", "South Africa", "South Korea", "South Sudan", "Spain", "Sri Lanka", "Sudan", "Suriname", "Sweden", "Switzerland", "Syria", "Taiwan", "Tajikistan", "Tanzania", "Thailand", "Timor-Leste", "Togo", "Tonga", "Trinidad and Tobago", "Tunisia", "Turkey", "Turkmenistan", "Tuvalu", "Uganda", "Ukraine", "United Arab Emirates", "United Kingdom", "United States", "Uruguay", "Uzbekistan", "Vanuatu", "Vatican City", "Venezuela", "Vietnam", "Yemen", "Zambia", "Zimbabwe"];
sort($countries);

function obfuscate_email($email) {
    if (!$email) return '';
    $parts = explode('@', $email);
    if (count($parts) !== 2) return $email;
    $name = substr($parts[0], 0, 2) . str_repeat('*', max(0, strlen($parts[0])-2));
    $domain = explode('.', $parts[1]);
    $domain_main = substr($domain[0], 0, 2) . str_repeat('*', max(0, strlen($domain[0])-2));
    $domain_ext = isset($domain[1]) ? $domain[1] : '';
    return $name . '@' . $domain_main . (strlen($domain_ext) ? ('.' . $domain_ext) : '');
}

$password_update_message = '';
if (isset($_POST['change_password']) && isset($_SESSION['username']) && $_SESSION['username'] === $user['username']) {
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';

    if (empty($new_password) || empty($confirm_password)) {
        $password_update_message = t('error_password_fields_required', $lang);
    } elseif ($new_password !== $confirm_password) {
        $password_update_message = t('error_passwords_no_match', $lang);
    } elseif (strlen($new_password) < 6) {
        $password_update_message = t('error_password_length', $lang);
    } else {
        $users = getUsers();
        foreach ($users as &$u) {
            if ($u['username'] === $_SESSION['username']) {
                $u['password'] = password_hash($new_password, PASSWORD_DEFAULT);
                break;
            }
        }
        saveUsers($users);
        $password_update_message = t('success_password_updated', $lang);
    }
}

$show_add_form = isset($_POST['show_add_form']) ? intval($_POST['show_add_form']) : -1;
// Remove password modal logic and add inline password editing
$show_password_field = isset($_POST['show_password_field']);

// Handle success messages from redirects
$success_message = $_GET['message'] ?? '';
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Profile - <?php echo htmlspecialchars($user['username']); ?> | <?php echo htmlspecialchars(getSiteName()); ?></title>
    <link rel="stylesheet" href="style.css" />
    <link rel="stylesheet" href="dynamic-styles.php" />
    <style>
        .profile-main {
            display: flex;
            flex-flow: column;
            gap: 20px;
            max-width: 400px;
            margin: 20px auto 0 auto;
            flex-wrap: wrap;
        }
        .profile-info-block {
            position: relative;
            /* flex: 0 0 320px; */
            background: #fff;
            border-radius: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.08);
            padding: 15px 0 15px 20px;
            min-width: 280px;
            max-width: 100%;
        }
        .profile-avatar {
            width: 90px;
            height: 90px;
            border-radius: 50%;
            background: <?php echo getAvatarColor($user['username']); ?>;
            color: #fff;
            font-size: 2.8rem;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 20px auto;
            font-family: 'Sniglet', Comic Sans MS, sans-serif;
            font-weight: bold;
        }
        .profile-username {
            font-size: 2rem;
            font-family: 'Sniglet', Comic Sans MS, sans-serif;
            margin-bottom: 10px;
        }
        .profile-meta {
            color: #888;
            font-size: 1rem;
            margin-bottom: 10px;
        }
        .profile-info-form label {
            display: block;
            margin-bottom: 4px;
            color: #555;
            font-weight: 500;
            margin-top: 12px;
        }
        .profile-info-form input, .profile-info-form select {
            width: 100%;
            padding: 10px;
            border-radius: 8px;
            border: 1px solid #eee;
            margin-bottom: 4px;
            font-size: 1.1rem;
        }
        .profile-info-form .change-password-btn {
            background: #ffe08f;
            color: #333;
            border: none;
            border-radius: 8px;
            padding: 8px 18px;
            font-size: 1rem;
            margin-top: 8px;
            cursor: pointer;
            transition: background 0.2s;
        }
        .profile-info-form .change-password-btn:hover {
            background: #f9a1b0;
            color: #fff;
        }
        .profile-info-form .save-btn {
            background: #7dcf95;
            color: #fff;
            border: none;
            border-radius: 8px;
            padding: 10px 24px;
            font-size: 1.1rem;
            margin-top: 18px;
            cursor: pointer;
            transition: background 0.2s;
        }
        .profile-info-form .save-btn:hover {
            background: #4a90e2;
        }
        .profile-info-form .readonly {
            background: #f7f7f7;
            color: #aaa;
        }
        .profile-info-form .joined-date {
            margin-top: 10px;
            color: #888;
            font-size: 1rem;
        }
        #obfuscated-email{
            position: absolute;
            bottom: 0px;  
            
        }
        #profile-value-email strong{
            margin-top: -10px; 
            
        }
        td#profile-value-username{
            width: 55%;
        }
        #profile-value-email{
            position: relative;
            height: 42px;
        }
        #toggleEmailBtn {
            padding-bottom: 0;
            margin-left: 0px;
            position: absolute;
            bottom: -5px;    
        }   
        .profile-principles-block {
            flex: 1 1 400px;
            min-width: 320px;
            background: #fff;
            border-radius: 20px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.08);
            padding: 28px 28px 28px 28px;
            margin-bottom: 30px;
            max-width: 100%;
        }
        .principles-grid {
            display: flex;
            flex-flow:column;
            gap: 15px;
        }
        .principle-card {
            background: #f7f7f7;
            border-radius: 16px;
            min-height: 310px;
            height: 310px;
            display: flex;
            flex-direction: column;
            box-shadow: 0 2px 8px rgba(0,0,0,0.04);
            padding: 18px 18px 18px 18px;
            position: relative;
            border: 1px solid lightgrey;
        }
        .principle-card.pending {
    margin-bottom: 10px;
}
        .principle-title {
            font-size: 0.95rem;
            font-family: 'Sniglet', Comic Sans MS, sans-serif;
            color: #4a90e2;
            margin-bottom: 6px;
            font-weight: bold;
        }
        .principle-desc {
            width: 100%;
            color: #444;
            font-size: 0.75rem;
            margin-bottom: 8px;
            margin-top: 5px;
            flex: 1;
            padding-right: 0px;
        }
        .principle-content {
            display: flex;
            flex-direction: column;
            height: 100%;
            width: 100%;
        }
        .principle-meta {
            position: absolute;
            right:15px;
            color: #888;
            font-size: 0.65rem;
            margin-bottom: 4px;
            display: flex;
            justify-content: space-between;
            align-items: flex-end;
        }
        .principle-action-btn {
            padding: 0 5px;
            background: none;
            border: none;
            cursor: pointer;
            font-size: 0.95rem;
            color: #888;
            transition: color 0.2s;
        }
        .principle-action-btn:hover {
            color: #e57373;
        }
        .moderation-status {
            display: flex;
            align-items: center;
            gap: 6px;
            color: #e28f00;
            font-size: 1rem;
            margin-top: 8px;
            padding-top: 6px;
        }
        .empty-slot {
            background: none;
            border: 2px dotted #4a90e2;
            border-radius: 16px;
            min-height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #4a90e2;
            font-size: 1.1rem;
            font-family: 'Sniglet', Comic Sans MS, sans-serif;
            flex-direction: column;
            gap: 8px;
            cursor: pointer;
            transition: background .2s;
        }
        .empty-slot:hover {
            background: #eafaff;
        }
        .empty-slot .empty-icon {
            font-size: 2.2rem;
            margin-bottom: 4px;
            color: #4a90e2;
        }
        .placeholder-card {
            width: 100%;
            height: 100%;
            background: none;
            border: 2px solid #e0e0e0;
            border-radius: 16px;
            min-height: 180px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #aaa;
            font-size: 1.1rem;
            font-family: 'Sniglet', Comic Sans MS, sans-serif;
            text-align: center;
            padding: 18px;
        }
        .principle-card.filled {
            background: #f7f7f7;
        }
        @media (max-width: 900px) {
            .profile-main {
                flex-direction: column;
                gap: 0;
                margin:0 5px;
            }
            .profile-info-block, .profile-principles-block {
                margin-bottom: 24px;
            }
            .profile-principles-block{
                padding: 30px 15px;
            }
            .principle-number-label{
                top: -25px !important;
            }
            .profile-info-block{
                padding: 0 10px 10px;
            }
        }
        .profile-info-form input[readonly] {
            background: #f7f7f7;
            cursor: pointer;
        }
        .add-principle-btn {
            background: #f9a1b0;
            color: #fff;
            border: none;
            border-radius: 30px;
            padding: 12px 28px;
            font-size: 1.1rem;
            font-family: 'Sniglet', Comic Sans MS, sans-serif;
            box-shadow: 0 2px 8px #f9a1b0;
            display: flex;
            align-items: center;
            gap: 10px;
            cursor: pointer;
            transition: background 0.2s;
        }
        .add-principle-btn:hover {
            background: #e68090;
        }
        .add-principle-btn .plus-icon {
            font-size: 1.4em;
            font-weight: bold;
        }
        .nav-container {
            width: 100%;
            max-width: 900px;
            margin: 0 auto;
        }
        .nav-left {
            display: flex;
            width: 100%;
            gap: 0;
        }
        .logout-btn {
            background: none;
            color: red;
            border: none;
            padding: 0 0 0 0px;
            font-size: 1.25rem;
            font-family: 'Sniglet', Comic Sans MS, sans-serif;
            font-weight: bold;
            cursor: pointer;
            /* transition: background 0.2s; */
        }
        .profile-field-row {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-bottom: 10px;
        }
        .profile-label {
            padding: 5px 0;
            font-size: 0.75rem;
            color: #555;
            font-weight: 900;
            text-align:left;
        }
        .profile-value {
            width: 25%;
            /* text-align:center; */
            /* padding: 5px 0; */
            /* font-size: 1.1rem; */
            font-size: 0.75rem;
            color: #333;
            /* flex-grow: 1; */
        }
        .profile-value strong{
            display:block;
            margin-top: -10px;
        }
        .edit-btn {
            background: none;
            border: none;
            cursor: pointer;
            font-size: 1.2rem;
            color: #888;
            transition: color 0.2s;
            /* display: none; //Hidden by default */
        }
        .edit-btn:hover {
            color: #4a90e2;
        }
        .profile-edit-input {
            width: 90%;
            padding: 4px;
            border-radius: 4px;
            border: 1px solid #eee;
            font-size: 0.65rem;
            box-sizing: border-box; /* Include padding and border in element's total width and height */
        }
        .edit-tick, .edit-cross {
            background: none;
            border: none;
            cursor: pointer;
            font-size: 1.2rem;
            color: #888;
            transition: color 0.2s;
        }
        .edit-tick:hover, .edit-cross:hover {
            color: #e57373;
        }
        .responsive-table-wrapper {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch; /* Smooth scrolling on iOS */
        }
        .profile-page .principle-title{
            padding-right:75px;
        }
    </style>
</head>
<body class="profile-page">
<nav class="navbar">
    <div class="nav-container nav-flex">
        <div class="nav-left">
            <a href="index.php" class="nav-button"><?php echo t('prescription', $lang); ?></a>
            <a href="principles.php" class="nav-button"><?php echo t('principles', $lang); ?></a>
            <a href="walkers.php" class="nav-button"><?php echo t('walkers', $lang); ?></a>
            <?php if (isLoggedIn()): ?>
                <a href="profile.php" class="nav-button profile-btn"><?php echo t('profile', $lang); ?></a>
            <?php else: ?>
                <a href="signup.php" class="nav-button login-btn"><?php echo t('login_signup', $lang); ?></a>
            <?php endif; ?>
        </div>
    </div>
</nav>
<div class="profile-main">
    <?php if ($success_message): ?>
        <div class="success-message" style="background-color: #e0f2f1; color: #00796b; padding: 15px; border-radius: 8px; margin-bottom: 20px; text-align: center; font-weight: bold;">
            <?php echo htmlspecialchars($success_message); ?>
        </div>
        <script>
            setTimeout(() => {
                const msg = document.querySelector('.success-message');
                if(msg) msg.style.display = 'none';
            }, 4000);
        </script>
    <?php endif; ?>
    <div class="profile-info-block">
        <form id="profileEditForm" method="post" style="margin-bottom:0;margin-top: 0px;">
            <div class="responsive-table-wrapper">
            <table class="profile-info-table" style="width:100%;border-collapse:separate;border-spacing:0 8px;">
                <thead>
                    <tr>
                        <th class="profile-label"><?php echo t('username', $lang); ?></th>
                        <th class="profile-label"><?php echo t('joined', $lang); ?></th>
                    </tr>
                </thead>
                <tbody id="profileFields">
                    <!-- First Row: Username and Joined Date -->
                    <tr>
                        <td class="profile-value" id="profile-value-username"><?php echo htmlspecialchars($user['username']); ?></td>
                        <td class="profile-value" id="profile-value-joined"><?php echo date('m/y', strtotime($user['joined'])); ?></td>
                    </tr>
                    <!-- Second Row: Email and Password -->
                    <tr>
                        <td class="profile-value" id="profile-value-email">
                            <strong><?php echo t('email', $lang); ?>:</strong>
                            <?php if ($is_own_profile): ?>
                                <?php echo htmlspecialchars($user['email']); ?>
                            <?php else: ?>
                                <?php if (!empty($user['email'])): ?>
                                    <span id="obfuscated-email" style="vertical-align: sub;"></span>
                                    <button type="button" id="toggleEmailBtn" class="profile-info-form save-btn" style="font-size:0.95rem;">👁️</button>
                                <?php endif; ?>
                            <?php endif; ?>
                        </td>
                        <?php if ($is_own_profile): ?>
                            <td class="profile-value" id="profile-value-password">
                                <strong><?php echo t('password', $lang); ?>:</strong> ••••••
                            </td>
                        <?php else: ?>
                            <td class="profile-value"></td>
                        <?php endif; ?>
                    </tr>
                </tbody>
            </table>
            </div>
            <?php if ($is_own_profile): ?>
            <div id="editProfileButtons" style="margin-top:0px;text-align:right;float:left;">
                <button type="button" id="editProfileBtn" class="profile-info-form edit-btn" title="<?php echo t('edit_profile', $lang); ?>"><?php echo t('icon_edit', $lang); ?></button>
                <button type="button" id="saveProfileBtn" class="profile-info-form save-btn" title="<?php echo t('save', $lang); ?>" style="display:none;"><?php echo t('icon_save', $lang); ?></button>
                <button type="button" id="cancelProfileBtn" class="profile-info-form cancel-btn" title="<?php echo t('cancel', $lang); ?>" style="display:none !important;"><?php echo t('icon_cancel', $lang); ?></button>
            </div>
            <?php endif; ?>
        </form>
        <?php if ($is_own_profile): ?>
        <form method="post" style="margin-top: 0px;display: inline-block;position: absolute;right: 20px;top: 5px;">
            <button type="submit" name="logout" class="logout-btn" title="<?php echo t('logout', $lang); ?>"><?php echo t('icon_logout', $lang); ?></button>
        </form>
        <?php endif; ?>
    </div>
    <div class="profile-principles-block">
        <?php if ($is_own_profile): ?>
        <div class="principles-header" style="display: flex; justify-content: center; align-items: center; margin-bottom: 10px; padding: 0 10px;">
            <div class="principles-counter" style="font-size: 0.65rem; color: #666; font-family: 'Sniglet', Comic Sans MS, sans-serif; font-weight: bold;">
                <?php echo t('principles', $lang); ?>: <?php echo count($user_principles); ?>/10
            </div>
            
        </div>
        <?php endif; ?>
        <div class="principles-grid" id="principlesGrid">
<?php 
// For public profiles, filter out non-approved principles
if (!$is_own_profile) {
    $user_principles = array_values(array_filter($user_principles, function($p) {
        return isset($p['approved']) && $p['approved'] === true;
    }));
}

$principle_count = count($user_principles);
$has_available_slots = $principle_count < 10;
$next_available_slot = $principle_count; // Index of next available slot

// For public profiles with no approved principles, show message
if (!$is_own_profile && $principle_count === 0) {
    echo '<div style="text-align: center; padding: 40px; color: #888; font-family: \'Sniglet\', Comic Sans MS, sans-serif; font-size: 1.1rem;">';
    echo t('no_principles', $lang);
    echo '</div>';
} else {
    // For own profile: show all principles + empty slot if available
    // For public profile: only show approved principles, no empty slots
    if ($is_own_profile) {
        $cards_to_show = $principle_count === 0 ? 1 : $principle_count + ($has_available_slots ? 1 : 0);
    } else {
        $cards_to_show = $principle_count; // Only show actual approved principles
    }
    
    for ($i = 0; $i < $cards_to_show; $i++):
        $p = $user_principles[$i] ?? null;
?>
    <?php if ($is_own_profile): ?>
        <div class="principle-card<?php echo ($p && empty($p['approved'])) ? ' pending' : ''; ?>" data-slot="<?php echo $i; ?>">
            <?php if ($p): ?>
                <?php if ($is_own_profile || (isset($p['approved']) && $p['approved'] === true)): ?>
                    <div class="principle-content" data-id="<?php echo $p['id']; ?>">
                        <div class="principle-header">
                            <div class="principle-title principle-title-view"> <?php echo htmlspecialchars($p['title']); ?> </div>
                            <div class="principle-meta test1">
                            <div>
                            <?php if ($is_own_profile && empty($p['approved'])): ?>
                                
                                    <span class="moderation-icon" title="Waiting for moderation">⏳</span>
                                <?php endif; ?>
                                <?php if ($is_own_profile): ?>
                                    <button class="principle-action-btn edit-principle-btn" data-id="<?php echo $p['id']; ?>" title="Edit">✏️</button>
                                    <form method="post" style="display:inline;">
                                        <input type="hidden" name="principle_action" value="delete">
                                        <input type="hidden" name="principle_id" value="<?php echo $p['id']; ?>">
                                        <button type="submit" class="principle-action-btn" title="Delete" onclick="return confirm('Delete this principle?');">🗑️</button>
                                    </form>
                                </div>
                                <?php endif; ?>
                                <span class="time-ago" title="<?php echo $p['created_at']; ?>"> <?php echo date('m/y', strtotime($p['created_at'])); ?> </span>
                            </div>
                        </div>
                        <div class="principle-desc principle-desc-view"> <?php echo nl2br(htmlspecialchars($p['description'])); ?> </div>
                        <?php if ($is_own_profile): ?>
                            <form method="post" class="edit-principle-form" style="display:none;margin-top:0;">
                                <input type="hidden" name="principle_action" value="edit">
                                <input type="hidden" name="principle_id" value="<?php echo $p['id']; ?>">
                                <input type="text" name="edit_title" maxlength="100" placeholder="Title (max 3 words)" required class="edit-title-input" style="margin-bottom:6px;width:72%;display:inline;">
                                <span class="editTitleWarning" style="color:#e57373;display:none;font-size:0.95em;">Max 3 words allowed</span>
                                <textarea name="edit_desc" maxlength="777" placeholder="Description (max 777 chars)" required class="edit-desc-input" style="width:100%;margin-top:5px;margin-bottom:6px;display:inline;"></textarea>
                                <span class="editDescCount" style="color:#888;font-size:0.75em;float:right;">0/777</span>
                                <button type="submit" class="profile-info-form save-btn" style="margin-right:8px;"><?php echo t('icon_save', $lang); ?></button>
                                <button type="button" class="profile-info-form save-btn cancel-edit-btn"><?php echo t('icon_cancel', $lang); ?></button>
                            </form>
                        <?php endif; ?>
                    </div>
                <?php else: ?>
                    <!-- This should not happen due to filtering, but safety check -->
                    <div class="placeholder-card"></div>
                <?php endif; ?>
            <?php else: ?>
                <!-- Empty slot - only show plus if it's the next available slot -->
                <?php if ($i === $next_available_slot && $has_available_slots): ?>
                    <div class="empty-slot-container">
                        <button class="plus-btn-full" data-slot="<?php echo $i; ?>">
                            <span class="empty-icon">+</span>
                        </button>
                        <form method="post" action="" class="add-principle-form" autocomplete="off" style="display: none;">
                            <input type="hidden" name="add_principle" value="1">
                            <div class="add-principle-title">
                                <input type="text" name="principle_title" maxlength="100" placeholder="Title (max 3 words)" required id="addPrincipleTitle">
                                <span id="addTitleWarning" style="color:#e57373;display:none;font-size:0.95em;">Max 3 words allowed</span>
                            </div>
                            <div class="add-principle-desc">
                                <textarea name="principle_desc" maxlength="777" placeholder="Description (max 777 chars)" required id="addPrincipleDesc"></textarea>
                                <span id="addDescCount" style="color:#888;font-size:0.75rem;float:right;">0/777</span>
                            </div>
                            <button type="submit" class="profile-info-form save-btn"><?php echo t('icon_save', $lang); ?></button>
                        </form>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    <?php else: ?>
        <!-- Other users' profile: only show approved principles -->
        <?php if ($p && isset($p['approved']) && $p['approved'] === true): ?>
        <div class="principle-card">
            <div class="principle-content" data-id="<?php echo $p['id']; ?>">
                <div class="principle-header">
                    <div class="principle-title principle-title-view"> <?php echo htmlspecialchars($p['title']); ?> </div>
                    <div class="principle-meta">
                        <span class="time-ago" title="<?php echo $p['created_at']; ?>"> <?php echo date('m/y', strtotime($p['created_at'])); ?> </span>
                    </div>
                </div>
                <div class="principle-desc principle-desc-view"> <?php echo nl2br(htmlspecialchars($p['description'])); ?> </div>
            </div>
        </div>
        <?php endif; ?>
    <?php endif; ?>
<?php endfor; ?>
<?php } // End of else block for principles display ?>
</div>
</div>
<script>
// Email reveal on click
const emailInput = document.getElementById('profile-email');
if (emailInput) {
    emailInput.addEventListener('click', function() {
        this.value = <?php echo json_encode($user['email']); ?>;
        this.style.cursor = 'default';
    }, { once: true });
}
// Close modal on outside click
document.addEventListener('click', function(e) {
    const modal = document.getElementById('changePasswordModal');
    if (modal && modal.style.display === 'flex' && e.target === modal) {
        modal.style.display = 'none';
    }
});
// Change password button triggers modal
const changePasswordBtn = document.querySelector('.change-password-btn');
if (changePasswordBtn) {
    changePasswordBtn.onclick = showChangePasswordModal;
}
// Character countdown script
const descTextarea = document.getElementById('principle_desc');
const charCountEl = document.getElementById('charCount');
if (descTextarea && charCountEl) {
    const maxLength = descTextarea.maxLength;
    const updateCount = () => {
        const remaining = maxLength - descTextarea.value.length;
        charCountEl.textContent = `${remaining} characters remaining`;
    };
    descTextarea.addEventListener('input', updateCount);
    updateCount(); // Initial count
}
// Auto-hide success messages after 5 seconds
setTimeout(function() {
    var updateMsg = document.getElementById('updateMessage');
    if (updateMsg) updateMsg.style.display = 'none';
    var passwordMsg = document.getElementById('passwordUpdateMessage');
    if (passwordMsg && passwordMsg.style.color === 'green') passwordMsg.style.display = 'none';
}, 5000);
// Enforce two-word limit for principle title
const titleInput = document.getElementById('principle_title');
const titleWarning = document.getElementById('titleWarning');
if (titleInput && titleWarning) {
    titleInput.addEventListener('input', function() {
        const words = this.value.trim().split(/\s+/);
        if (words.length > 2) {
            titleWarning.style.display = 'block';
            this.value = words.slice(0,2).join(' ');
        } else {
            titleWarning.style.display = 'none';
        }
    });
}
// Save scroll position before submitting the plus icon form
const plusForms = document.querySelectorAll('form input[name="show_add_form"]');
plusForms.forEach(input => {
    input.form.addEventListener('submit', function(e) {
        localStorage.setItem('profileScrollY', window.scrollY);
    });
});
// Restore scroll position on page load
window.addEventListener('DOMContentLoaded', function() {
    const y = localStorage.getItem('profileScrollY');
    if (y !== null) {
        window.scrollTo(0, parseInt(y));
        localStorage.removeItem('profileScrollY');
    }
});


// Add JS for edit mode
const editProfileBtn = document.getElementById('editProfileBtn');
const profileFields = document.getElementById('profileFields');
const editProfileButtons = document.getElementById('editProfileButtons');
const logoutButton = document.getElementsByClassName('logout-btn');
let originalValues = {};

if (editProfileBtn) {
    editProfileBtn.addEventListener('click', function() {
        // Save original values
        originalValues = {
            username: document.getElementById('profile-value-username').textContent,
            email: document.getElementById('profile-value-email').textContent,
            joined: document.getElementById('profile-value-joined').textContent,
            password: document.getElementById('profile-value-password').textContent // Keep password as is
        };
        // Extract clean email value (remove the "Email:" label if present)
        let emailValue = originalValues.email;
        // Remove "Email:" prefix and any whitespace, get only the actual email
        emailValue = emailValue.replace(/^Email:\s*/, '').trim();
        // If there are still unwanted characters, extract just the email part
        const emailMatch = emailValue.match(/[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/);
        if (emailMatch) {
            emailValue = emailMatch[0];
        } else {
            // If no valid email found or it's just the label, set to empty
            emailValue = '';
        }
        
        // Replace with two-row layout structure to match the original
        profileFields.innerHTML = `
            <!-- First Row: Username and Joined Date -->
            <tr>
                <td class='profile-value'><input class="profile-edit-input" type='text' name='username' value='${originalValues.username}' maxlength='32' required></td>
                <td class='profile-value'>${originalValues.joined}</td>
            </tr>
            <!-- Second Row: Email and Password -->
            <tr>
                <td class='profile-value'>
                    <strong>Email:</strong><br>
                    <input class="profile-edit-input" type='email' name='email' value='${emailValue}' maxlength='64'>
                </td>
                <td class='profile-value'>
                    <strong>Password:</strong><br>
                    <input class="profile-edit-input" type='password' name='new_password' id='new_password' placeholder='New password' style='margin-bottom:4px;'><br>
                    <input class="profile-edit-input" type='password' name='confirm_password' id='confirm_password' placeholder='Confirm new password'><br>
                    <div id='password_error' style='color:red;font-size:0.9em;margin-top:4px;display:none;'></div>
                </td>
            </tr>
        `;
        // Hide edit button, show save/cancel buttons
        document.getElementById('editProfileBtn').style.display = 'none';
        document.getElementsByClassName('logout-btn')[0].style.display = 'none';
        document.getElementById('saveProfileBtn').style.display = 'inline-block';
        document.getElementById('cancelProfileBtn').style.display = 'none';
        
        // Handle save button click
        document.getElementById('saveProfileBtn').addEventListener('click', function(e) {
            const newPassword = document.getElementById('new_password').value;
            const confirmPassword = document.getElementById('confirm_password').value;
            
            if (newPassword && newPassword !== confirmPassword) {
                e.preventDefault();
                document.getElementById('password_error').textContent = '<?php echo t('password_mismatch', $lang); ?>';
                document.getElementById('password_error').style.display = 'block';
                return false;
            } else {
                // Passwords match or no password change, submit the form
                document.getElementById('password_error').style.display = 'none';
                // Find the parent form and submit it
                const form = document.querySelector('form');
                if (form) {
                    form.submit();
                }
            }
        });
        // Add password confirmation validation
        const newPasswordField = document.getElementById('new_password');
        const confirmPasswordField = document.getElementById('confirm_password');
        const passwordError = document.getElementById('password_error');
        
        function validatePasswords() {
            const newPassword = newPasswordField.value;
            const confirmPassword = confirmPasswordField.value;
            
            if (confirmPassword && newPassword !== confirmPassword) {
                passwordError.textContent = '<?php echo t('password_mismatch', $lang); ?>';
                passwordError.style.display = 'block';
                return false;
            } else {
                passwordError.style.display = 'none';
                return true;
            }
        }
        
        newPasswordField.addEventListener('input', validatePasswords);
        confirmPasswordField.addEventListener('input', validatePasswords);
        
        // Add cancel button functionality
        document.getElementById('cancelProfileBtn').addEventListener('click', function() {
            // Restore original view
            profileFields.innerHTML = `
                <tr>
                    <td class='profile-value' id='profile-value-username'>${originalValues.username}</td>
                    <td class='profile-value' id='profile-value-joined'>${originalValues.joined}</td>
                    <td class='profile-value' id='profile-value-email'>${originalValues.email}</td>
                    <?php if ($is_own_profile): ?>
                        <td class='profile-value' id='profile-value-password'>••••••</td>
                    <?php endif; ?>
                </tr>
            `;
            // Show edit button, hide save/cancel buttons
            document.getElementById('editProfileBtn').style.display = 'inline-block';
            document.getElementById('saveProfileBtn').style.display = 'none';
            document.getElementById('cancelProfileBtn').style.display = 'none';
        });
    });
}
// Email view/hide logic for other users
<?php if (!$is_own_profile): ?>
const toggleEmailBtn = document.getElementById('toggleEmailBtn');
const obfuscatedEmail = document.getElementById('obfuscated-email');
if (toggleEmailBtn && obfuscatedEmail) {
    const realEmail = <?php echo json_encode($user['email']); ?>;
    toggleEmailBtn.addEventListener('click', function() {
        obfuscatedEmail.textContent = realEmail;
        toggleEmailBtn.style.display = 'none';
    });
}
<?php endif; ?>

// Corrected JavaScript for showing the add form
document.querySelectorAll('.plus-btn-full').forEach(btn => {
    btn.addEventListener('click', function(e) {
        // We no longer need e.preventDefault() here because the button is not a submit type
        const container = btn.closest('.empty-slot-container');
        btn.style.display = 'none'; // Hide the plus button
        const form = container.querySelector('.add-principle-form');
        if (form) {
            form.style.display = 'block'; // Show the form
            form.querySelector('input[name="principle_title"]').focus();
        }
    });
});

// Add Principle: enforce 3-word title and 777-char desc
const addTitleInput = document.getElementById('addPrincipleTitle');
const addTitleWarning = document.getElementById('addTitleWarning');
const addDescInput = document.getElementById('addPrincipleDesc');
const addDescCount = document.getElementById('addDescCount');
if (addTitleInput && addTitleWarning) {
    addTitleInput.addEventListener('input', function() {
        const words = this.value.trim().split(/\s+/);
        if (words.length > 3) {
            addTitleWarning.style.display = 'inline';
            this.value = words.slice(0,3).join(' ');
        } else {
            addTitleWarning.style.display = 'none';
        }
    });
}
if (addDescInput && addDescCount) {
    addDescInput.addEventListener('input', function() {
        const currentLength = this.value.length;
        addDescCount.textContent = currentLength + '/777';
        if (this.value.length > 777) {
            this.value = this.value.slice(0,777);
            addDescCount.textContent = '777/777';
        }
    });
    // Initialize counter
    addDescCount.textContent = '0/777';
}

// Function to show add principle form when all slots are filled
function showAddPrincipleForm() {
    // Find the first empty slot and show its form
    const emptySlots = document.querySelectorAll('.empty-slot-container');
    if (emptySlots.length > 0) {
        const firstEmptySlot = emptySlots[0];
        const plusBtn = firstEmptySlot.querySelector('.plus-btn-full');
        const form = firstEmptySlot.querySelector('.add-principle-form');
        if (plusBtn && form) {
            plusBtn.style.display = 'none';
            form.style.display = 'block';
            form.querySelector('input[name="principle_title"]').focus();
        }
    }
}
// Inline edit for principle
function setupInlineEditPrinciple() {
    document.querySelectorAll('.edit-principle-btn').forEach(function(btn) {
        const content = btn.closest('.principle-content');
        const titleView = content.querySelector('.principle-title-view');
        const descView = content.querySelector('.principle-desc-view');
        const form = content.querySelector('.edit-principle-form');
        const header = btn.closest(".principle-header");
        if (!titleView || !descView || !form) return;
        btn.onclick = function(e) {
            e.preventDefault();
            if (header) {
                header.classList.add("expanded");
            }   
            // Save original text
            const origTitle = titleView.textContent.trim();
            const origDesc = descView.textContent.trim();
            // Hide view, show form
            titleView.style.display = 'none';
            descView.style.display = 'none';
            form.style.display = 'block';
            // Place form inline (replace the text nodes)
            titleView.parentNode.insertBefore(form, titleView.nextSibling);
            // Fill form fields
            form.querySelector('.edit-title-input').value = origTitle;
            form.querySelector('.edit-desc-input').value = origDesc;
            // Live 3-word and char limit
            const editTitleInput = form.querySelector('.edit-title-input');
            const editTitleWarning = form.querySelector('.editTitleWarning');
            editTitleInput.addEventListener('input', function() {
                const words = this.value.trim().split(/\s+/);
                if (words.length > 3) {
                    editTitleWarning.style.display = 'inline';
                    this.value = words.slice(0,3).join(' ');
                } else {
                    editTitleWarning.style.display = 'none';
                }
            });
            const editDescInput = form.querySelector('.edit-desc-input');
            const editDescCount = form.querySelector('.editDescCount');
            editDescInput.addEventListener('input', function() {
                const currentLength = this.value.length;
                editDescCount.textContent = currentLength + '/777';
                if (this.value.length > 777) {
                    this.value = this.value.slice(0,777);
                    editDescCount.textContent = '777/777';
                }
            });
            // Initialize counter
            editDescCount.textContent = editDescInput.value.length + '/777';
            // Cancel button
            form.querySelector('.cancel-edit-btn').onclick = function() {
                form.style.display = 'none';
                titleView.style.display = '';
                descView.style.display = '';
                if (header) {
                    header.classList.remove("expanded");
                }   
            };
        };
    });
}
setupInlineEditPrinciple();

</script>
</body>
</html> 