<?php
session_start();

// Ensure settings.php is loaded for getText() function
require_once __DIR__ . '/settings.php';


/**
 * Translation function - ALIAS for getText() from settings.php
 * This function maintains backward compatibility while using the new settings system.
 * 
 * @param string $key The translation key
 * @param string $lang The language code (optional, will use current language)
 * @return string The translated text or the key if not found
 */
function t($key, $lang = null) {
    // Use getAppText() function from settings.php
    // Renamed to avoid conflict with PHP's built-in gettext()
    if (function_exists('getAppText')) {
        return getAppText($key, $lang);
    }
    // Fallback if getAppText is not available
    return $key;
}

/**
 * Principle translations for demo content
 * Add your own principle translations here
 */
$PRINCIPLE_TRANSLATIONS = [
    'Resilience in Adversity' => [
        'fr' => [
            'title' => 'Résilience dans l\'adversité',
            'description' => 'Quand la vie devient difficile, nous devons nous accrocher plus fort à notre but. La résilience n\'est pas de ne jamais tomber, mais de se relever à chaque fois.'
        ]
    ],
    // Add more principle translations as needed
];

/**
 * Translate principle content
 * 
 * @param array $principle The principle array
 * @param string $lang The language code
 * @return array The translated principle
 */
function translate_principle($principle, $lang) {
    global $PRINCIPLE_TRANSLATIONS;
    if ($lang === 'en') return $principle;
    $t = $PRINCIPLE_TRANSLATIONS[$principle['title']] ?? null;
    if ($t && isset($t[$lang])) {
        return [
            'title' => $t[$lang]['title'],
            'description' => $t[$lang]['description'] ?? $principle['description'],
            'created_at' => $principle['created_at']
        ];
    }
    return $principle;
}

/**
 * Get all users from the JSON file
 * 
 * @return array Array of user objects
 */
function getUsers() {
    $jsonFile = __DIR__ . '/users.json';
    if (file_exists($jsonFile)) {
        $jsonData = file_get_contents($jsonFile);
        $users = json_decode($jsonData, true);
        // Ensure it returns an array even if the file is empty or corrupt
        return is_array($users) ? $users : [];
    }
    return [];
}

/**
 * Save users to the JSON file
 * 
 * @param array $users Array of user objects
 */
function saveUsers($users) {
    $jsonFile = __DIR__ . '/users.json';
    file_put_contents($jsonFile, json_encode($users, JSON_PRETTY_PRINT));
}

/**
 * Register a new user
 * 
 * @param string $username The username
 * @param string $email The email address
 * @param string $password The password
 * @return bool True if successful, false if username/email already exists
 */
function registerUser($username, $email, $password) {
    $users = getUsers();
    
    // Check if username already exists
    foreach ($users as $user) {
        if ($user['username'] === $username) {
            return ['success' => false, 'message' => 'Username already exists'];
        }
    }
    // (No email uniqueness check)
    // Generate unique user_id
    $user_id = generateUniqueUserId();
    
    $newUser = [
        'user_id' => $user_id,
        'username' => $username,
        'email' => $email,
        'password' => password_hash($password, PASSWORD_DEFAULT),
        'joined' => date('Y-m-d H:i:s'),
        'is_moderator' => false
    ];
    
    $users[] = $newUser;
    saveUsers($users);
    
    return ['success' => true, 'message' => t('success_user_registered', getCurrentLanguage())];
}

/**
 * Login a user
 * 
 * @param string $username The username
 * @param string $password The password
 * @return bool True if login successful, false otherwise
 */
function loginUser($username, $password) {
    $users = getUsers();
    
    foreach ($users as $user) {
        if ($user['username'] === $username && password_verify($password, $user['password'])) {
            $_SESSION['username'] = $username;
            return true;
        }
    }
    
    return false;
}

/**
 * Get user by username
 * 
 * @param string $username The username to search for
 * @return array|null User object or null if not found
 */
function getUserByUsername($username) {
    $users = getUsers();
    foreach ($users as $user) {
        if ($user['username'] === $username) {
            return $user;
        }
    }
    return null;
}

/**
 * Get user by email
 * 
 * @param string $email The email to search for
 * @return array|null User object or null if not found
 */
function getUserByEmail($email) {
    $users = getUsers();
    foreach ($users as $user) {
        if (isset($user['email']) && strtolower($user['email']) === strtolower($email)) {
            return $user;
        }
    }
    return null;
}

/**
 * Check if user is logged in
 * 
 * @return bool True if user is logged in
 */
function isLoggedIn() {
    return isset($_SESSION['username']);
}

/**
 * Check if current user is a moderator
 * Now uses the consolidated settings system
 * 
 * @return bool True if user is a moderator
 */
function isModerator() {
    if (!isLoggedIn()) return false;
    
    // Get current user to access user_id
    $user = getUserByUsername($_SESSION['username']);
    if (!$user || !isset($user['user_id'])) return false;
    
    // Use the consolidated settings system with user_id
    if (function_exists('isModeratorBySettings')) {
        return isModeratorBySettings($user['user_id']);
    }
    
    // Fallback to old system if settings function not available
    return $user && isset($user['is_moderator']) && $user['is_moderator'];
}

/**
 * Logout the current user
 */
function logoutUser() {
    session_destroy();
}

/**
 * Generate avatar color based on username
 * 
 * @param string $username The username
 * @return string Hex color code
 */
function getAvatarColor($username) {
    if (!$username || !is_string($username) || !isset($username[0])) {
        return '#90a4ae'; // A default neutral color
    }
    $colors = ['#7dcf95','#d29ec5','#7dcae6','#fdac62','#dcb35a','#e9734f','#4eafb8','#ed9ba6','#e18f67','#aac2aa','#7da3ba','#d29f9b','#9ca2c9','#c8b7ad','#86b8c1'];
    return $colors[ord(strtolower($username[0])) % count($colors)];
}

/**
 * Update user profile information
 * 
 * @param string $username The username
 * @param string $age The age
 * @param string $location The location
 * @return bool True if successful
 */
function updateUserProfile($username, $age, $location) {
    $users = getUsers();
    foreach ($users as &$user) {
        if ($user['username'] === $username) {
            $user['age'] = $age;
            $user['location'] = $location;
            break;
        }
    }
    saveUsers($users);
    return true;
}

/**
 * Get default language based on browser settings
 * Now uses the consolidated settings system
 * 
 * @return string Language code (default: 'en')
 */
function get_default_lang() {
    // Use the getCurrentLanguage() function from settings.php
    if (function_exists('getCurrentLanguage')) {
        return getCurrentLanguage();
    }
    
    // Fallback if getCurrentLanguage is not available
    if (function_exists('getSetting')) {
        return getSetting('LANGUAGES.default', 'en');
    }
    
    // Final fallback
    return 'en';
} 

/**
 * Generate a unique 10-digit user_id
 */
function generateUniqueUserId() {
    $users = getUsers();
    $existing_ids = array_column($users, 'user_id');
    
    do {
        $user_id = str_pad(rand(1000000000, 9999999999), 10, '0', STR_PAD_LEFT);
    } while (in_array($user_id, $existing_ids));
    
    return $user_id;
}

/**
 * Check if user_id exists
 */
function userIdExists($user_id) {
    $users = getUsers();
    foreach ($users as $user) {
        if (isset($user['user_id']) && $user['user_id'] === $user_id) {
            return true;
        }
    }
    return false;
}

/**
 * Get user by user_id
 */
function getUserByUserId($user_id) {
    $users = getUsers();
    foreach ($users as $user) {
        if (isset($user['user_id']) && $user['user_id'] === $user_id) {
            return $user;
        }
    }
    return null;
}

/**
 * Migrate existing data to use user_id
 */
function migrateToUserId() {
    // Add user_id to existing users
    $users = getUsers();
    $username_to_userid = [];
    
    foreach ($users as &$user) {
        if (!isset($user['user_id'])) {
            $user['user_id'] = generateUniqueUserId();
            $username_to_userid[$user['username']] = $user['user_id'];
        }
    }
    
    saveUsers($users);
    
    // Update principles to use user_id instead of username
    if (file_exists('principles.json')) {
        $principles = json_decode(file_get_contents('principles.json'), true);
        if (is_array($principles)) {
            foreach ($principles as &$principle) {
                if (isset($principle['posted_by']) && isset($username_to_userid[$principle['posted_by']])) {
                    $principle['posted_by_user_id'] = $username_to_userid[$principle['posted_by']];
                    // Keep posted_by for backward compatibility during transition
                }
            }
            file_put_contents('principles.json', json_encode($principles, JSON_PRETTY_PRINT));
        }
    }
    
    return $username_to_userid;
} 

function require_login() {
    if (!isset($_SESSION['username'])) {
        header('Location: signup.php?message=Please+log+in+to+continue');
        exit;
    }
} 